package propagator;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import fr.cnes.genius.exception.GException;
import fr.cnes.genius.exception.GFileManipulatorException;
import fr.cnes.genius.loggers.GConsoleLogger;
import fr.cnes.genius.main.GFileManipulation;
import fr.cnes.genius.sqlite.ColumnInfo;
import fr.cnes.genius.sqlite.ColumnInfo.ColumnType;
import fr.cnes.genius.sqlite.MadonaWriter;
import fr.cnes.genopus.exception.GPOrbitException;
import fr.cnes.genopus.exception.GPVehicleException;
import fr.cnes.sirius.addons.patriusdataset.PatriusDataset;
import fr.cnes.sirius.patrius.assembly.Vehicle;
import fr.cnes.sirius.patrius.assembly.models.MassModel;
import fr.cnes.sirius.patrius.attitudes.AttitudeLaw;
import fr.cnes.sirius.patrius.attitudes.LofOffset;
import fr.cnes.sirius.patrius.forces.ForceModel;
import fr.cnes.sirius.patrius.forces.ForceModelsData;
import fr.cnes.sirius.patrius.frames.FramesFactory;
import fr.cnes.sirius.patrius.frames.LOFType;
import fr.cnes.sirius.patrius.math.geometry.euclidean.threed.RotationOrder;
import fr.cnes.sirius.patrius.math.ode.FirstOrderIntegrator;
import fr.cnes.sirius.patrius.math.ode.nonstiff.ClassicalRungeKuttaIntegrator;
import fr.cnes.sirius.patrius.math.util.FastMath;
import fr.cnes.sirius.patrius.orbits.Orbit;
import fr.cnes.sirius.patrius.orbits.OrbitType;
import fr.cnes.sirius.patrius.orbits.PositionAngle;
import fr.cnes.sirius.patrius.orbits.orbitalparameters.CartesianParameters;
import fr.cnes.sirius.patrius.orbits.orbitalparameters.ReentryParameters;
import fr.cnes.sirius.patrius.orbits.pvcoordinates.PVCoordinates;
import fr.cnes.sirius.patrius.propagation.MassProvider;
import fr.cnes.sirius.patrius.propagation.SpacecraftState;
import fr.cnes.sirius.patrius.propagation.numerical.NumericalPropagator;
import fr.cnes.sirius.patrius.propagation.sampling.PatriusFixedStepHandler;
import fr.cnes.sirius.patrius.time.AbsoluteDate;
import fr.cnes.sirius.patrius.time.TimeScale;
import fr.cnes.sirius.patrius.time.TimeScalesFactory;
import fr.cnes.sirius.patrius.utils.Constants;
import fr.cnes.sirius.patrius.utils.exception.PatriusException;
import fr.cnes.sirius.patrius.utils.exception.PropagationException;

/**
 * Class corresponding to the batch computation mode
 * @author goesterjf
 *
 */
public class BatchPropagator {
    
    /** Initial orbit */
    private Orbit orbit;
    /** Vehicle characteristics */
    private Vehicle vehicle;
    /** Forces */
    private ForceModelsData forces;
    
    /** UTC time scale */
    private TimeScale UTC;
    
    /** By default EPHEM file names */
    private static final String EPH_FILE = "EPHEM.txt";

    /**
   * Constructor.
   * @param nomFicData    name of the context file.
   * @param nomFicEphem   name of the output file.
   * @throws IOException  GENIUS exception.
     * @throws PatriusException 
     */
    public BatchPropagator (final String nomFicData, final String nomFicEphem) throws IOException, PatriusException {
        
        // Patrius dataset initialization
        PatriusDataset.addResourcesFromPatriusDataset();
 
        WidPropagatorDataPanel dataPan = null;
        try {
            dataPan = new WidPropagatorDataPanel();
        } catch (GException err) {
            GConsoleLogger.getLogger().log(Level.SEVERE, err.getMessage());
        }           
        // Data read inside the XML file
        try {
            GFileManipulation.readConfig(nomFicData, "Propagator", dataPan, true);
        } catch (GFileManipulatorException err) {
            GConsoleLogger.getLogger().log(Level.SEVERE, err.getMessage());
        }            
        // Orbit initialization
        try {
            orbit = dataPan.getOrbit();
        } catch (GPOrbitException err) {
            GConsoleLogger.getLogger().log(Level.SEVERE, err.getMessage());
        }
        // Vehicle initialization
        try {
            vehicle = dataPan.getVehicle();
        } catch (GPVehicleException err) {
            GConsoleLogger.getLogger().log(Level.SEVERE, err.getMessage());
        }
        // Forces initialization
        try {
            forces = dataPan.getForces();
        } catch (GException err) {
            GConsoleLogger.getLogger().log(Level.SEVERE, err.getMessage());
        } catch (PatriusException err) {
            GConsoleLogger.getLogger().log(Level.SEVERE, err.getMessage());
        }
        
        // Recovery of the UTC time scale using a "factory"
        try {
            UTC = TimeScalesFactory.getUTC();
        } catch (PatriusException err) {
            GConsoleLogger.getLogger().log(Level.SEVERE, err.getMessage());
        }
 
    }
    
    /**
     * Method to propagate
     * @throws PatriusException 
     */
    public void compute() throws PatriusException {
        
        // Getting the mas provider from the vehicle object.
        final MassProvider mm = new MassModel(vehicle.createAssembly(FramesFactory.getCIRF()));
        
        // Initialization of the Runge Kutta integrator with a 5 s step
        final double pasRk = 5.;
        final FirstOrderIntegrator integrator = new ClassicalRungeKuttaIntegrator(pasRk);

        // Initialization of the propagator
        final NumericalPropagator propagator = new NumericalPropagator(integrator, orbit.getFrame(), OrbitType.CARTESIAN, PositionAngle.TRUE);
        
        SpacecraftState iniState = null;
        if ( mm.getTotalMass() <= 0. ) {
            iniState = new SpacecraftState(orbit);
        } else {
            iniState = new SpacecraftState(orbit, mm);
            // Adding additional state
            propagator.setMassProviderEquation(mm);
        }
        propagator.resetInitialState(iniState);
        
        // Adding an attitude law (in case of lift component)
        final AttitudeLaw attitudeLaw = new LofOffset(LOFType.LVLH, RotationOrder.ZYX, 0., 0., 0.);
        propagator.setAttitudeProvider(attitudeLaw);
        
        // Adding force models
        List<ForceModel> list = forces.getForceModelsList();
        for (ForceModel forceModel : list) {
            propagator.addForceModel(forceModel);            
        }
        
        // Creation of a fixed step handler
        final ArrayList<SpacecraftState> listOfStates = new ArrayList<SpacecraftState>();
        PatriusFixedStepHandler myStepHandler = new PatriusFixedStepHandler() {
            private static final long serialVersionUID = 1L;
            public void init(SpacecraftState s0, AbsoluteDate t) {
                // Nothing to do ...
            }
            public void handleStep(SpacecraftState currentState, boolean isLast)
                    throws PropagationException {
                // Adding S/C to the list
                listOfStates.add(currentState);
            }
        };
        // The handler frequency is set to 60s
        propagator.setMasterMode(60., myStepHandler);

        // Propagating on 1 period
        final double dt = orbit.getKeplerianPeriod();
        final AbsoluteDate finalDate = orbit.getDate().shiftedBy(dt);
        final SpacecraftState finalState = propagator.propagate(finalDate);
        final Orbit finalOrbit = finalState.getOrbit();
        
        writeEphem(finalOrbit, listOfStates);
        
    }
    
    private void writeEphem ( final Orbit finalOrbit, final ArrayList<SpacecraftState> listOfStates ) throws PatriusException {
        
        // Printing new date and semi major axis
        System.out.println();
        System.out.println("Initial date = "+orbit.getDate().toString(UTC));
        System.out.println("Initial semi major axis = "+orbit.getA()/1000.+" km");
        System.out.println("New date = "+finalOrbit.getDate().toString(UTC));
        System.out.println("Final semi major axis = "+finalOrbit.getA()/1000.+" km");
        
        // Writing in the EPHEM.txt file
//        try {
//            System.out.println();
//            System.out.println("EPHEM.txt file writing ...");
//            FileWriter ephem = new FileWriter(new File(EPH_FILE));
//            Locale.setDefault(Locale.US);
//            for (SpacecraftState sc : listOfStates) {
//                ephem.write(String.format("%s %22.15e \n", sc.getDate(), sc.getA()/1000.));
//            }
//            ephem.close();
//        } catch (IOException err) {
//            GConsoleLogger.getLogger().log(Level.SEVERE, err.getMessage());
//        }
        
        // Header information
        ArrayList<String> headerInfoLines = new ArrayList<String>();
        headerInfoLines.add("Logiciel=\"TEST\"");
        headerInfoLines.add("VERSION=\"Vx.x\"");
        
        // Initialization
        final MadonaWriter madonaWriter = new MadonaWriter(headerInfoLines);
        try {
            madonaWriter.createFile(new File(EPH_FILE));
        } catch (IOException e) {
            e.printStackTrace();
        }
        
        // Column information
        final ArrayList<ColumnInfo> columnInfoList =  new ArrayList<ColumnInfo>();
        ColumnInfo infoDate = new ColumnInfo("DATE", "Absolute date", ColumnType.DATE, "cal", null, true);
        ColumnInfo infoSma  = new ColumnInfo("SMA", "Semi major axis", ColumnType.REAL, "km", null, true);
        ColumnInfo infoAlt  = new ColumnInfo("ALT", "Altitude", ColumnType.REAL, "km", null, true);
        ColumnInfo infoLat  = new ColumnInfo("LAT", "Latitude", ColumnType.REAL, "deg", null, true);
        ColumnInfo infoLon  = new ColumnInfo("LON", "Longitude", ColumnType.REAL, "deg", null, true);
        columnInfoList.add(infoDate);
        columnInfoList.add(infoSma);
        columnInfoList.add(infoAlt);
        columnInfoList.add(infoLat);
        columnInfoList.add(infoLon);
        
        // Storing data in lists
        final ArrayList<Object> dateValues = new ArrayList<Object>();
        final ArrayList<Object> smaValues  = new ArrayList<Object>();
        final ArrayList<Object> altValues  = new ArrayList<Object>();
        final ArrayList<Object> latValues  = new ArrayList<Object>();
        final ArrayList<Object> lonValues  = new ArrayList<Object>();
        final double MU = Constants.WGS84_EARTH_MU;
        final double REQ  = Constants.WGS84_EARTH_EQUATORIAL_RADIUS;
        final double FLAT = Constants.WGS84_EARTH_FLATTENING;
        for (SpacecraftState sc : listOfStates) {
            dateValues.add(sc.getDate());
            smaValues.add(sc.getA()/1000.);
            final PVCoordinates pv = sc.getPVCoordinates(FramesFactory.getITRF());
            final CartesianParameters car = new CartesianParameters(pv, MU);
            final ReentryParameters ren = car.getReentryParameters(REQ, FLAT);
            altValues.add(ren.getAltitude()/1000.);
            latValues.add(FastMath.toDegrees(ren.getLatitude()));
            lonValues.add(FastMath.toDegrees(ren.getLongitude()));
        }
        
        // Adding columns
        madonaWriter.addColumns(infoDate, dateValues, 0);
        madonaWriter.addColumns(infoSma, smaValues, 1);
        madonaWriter.addColumns(infoAlt, altValues, 2);
        madonaWriter.addColumns(infoLat, latValues, 3);
        madonaWriter.addColumns(infoLon, lonValues, 4);
        
        // Storing data in file
        madonaWriter.writeHeader(columnInfoList);;
        madonaWriter.writeColumns();
        madonaWriter.close();

    }

    /**
     * Main method ...
     * @param args
     */
    public static void main(String[] args) {
        
        BatchPropagator batch;
        try {
            batch = new BatchPropagator("src/test/resources/INI_Propagator.xml", "EPHEM.txt");
            batch.compute();
        } catch (IOException err) {
            GConsoleLogger.getLogger().log(Level.SEVERE, err.getMessage());
        } catch (PropagationException err) {
            GConsoleLogger.getLogger().log(Level.SEVERE, err.getMessage());
        } catch (PatriusException err) {
            GConsoleLogger.getLogger().log(Level.SEVERE, err.getMessage());
        }
 
    }

}
