package propagator;

import java.awt.Color;
import java.io.File;
import java.util.ArrayList;

import fr.cnes.genius.chart.GCoordinatesData;
import fr.cnes.genius.exception.GException;
import fr.cnes.genius.exception.GFileManipulatorException;
import fr.cnes.genius.exception.GPlotDataReaderException;
import fr.cnes.genius.main.GFileManipulation;
import fr.cnes.genius.plots.GPlotDataMadonaReader;
import fr.cnes.genius.savefiles.GSaveResults;
import fr.cnes.genius.standardapplication.GAboutDialog;
import fr.cnes.genius.standardapplication.GMainFrameAbstract;
import fr.cnes.genius.threads.GCommandLauncher.ProcessStatus;
import fr.cnes.genius.utils.GContextFileManagement;
import fr.cnes.genius.utils.GFileFilter;
import fr.cnes.sirius.addons.patriusdataset.PatriusDataset;
import fr.cnes.sirius.patrius.utils.exception.PatriusException;

/**
 * Class defining the main frame.
 * @author goesterjf
 *
 */
public class WidPropagator extends GMainFrameAbstract<WidPropagatorDataPanel> {
    
    // FILE NAMES
    
    /** Prefix for context file names */
    private static final String INI_FILE_PREFIX = "INI_";
    /** By default context file names */
    private static final String INI_FILE = "INIT.xml";
    /** Prefix for output file names */
    private static final String EPH_FILE_PREFIX = "EPH_";
    /** By default EPHEM file names */
    private static final String EPH_FILE = "EPHEM.txt";
    
    // SIZES
    
    /** Data panel height */
    private static final int DATAPANEL_HEIGHT = 400;
    /** Error console height */
    private static final int ERRCONSOLE_HEIGHT = 80;
    /** Icon size */
    private static  final int ICON_SIZE = 12;
    
    /**
     * Constructor.
     * @throws GException   GENIUS exception
     */
    public WidPropagator() throws GException {
        
    super("Propagator",
          new WidPropagatorDataPanel(),
          new GContextFileManagement(".", "Propagator", new GFileFilter(INI_FILE_PREFIX, ".xml", "Propagator Files") ),
          new GAboutDialog("About S/W", "Propagator using PATRIUS ...", "CNES", "V1.0 ; 20/12/2017", null),
          new GSaveResults("Saving results", new File("."), ".txt", ".xml"),
          DATAPANEL_HEIGHT, ERRCONSOLE_HEIGHT, ICON_SIZE, false);
 
    }

    /**
     * Method to manage saving data and results files
     */
    @Override
    protected void saveFilesManagement() throws GException {

        final File ini = new File(INI_FILE);
        final File res = new File(EPH_FILE);
 
        // The context file INIT.xml will be saved in INI_....xml
        this.getSaveResultsDialog().setContextFile(ini.getName(), INI_FILE_PREFIX, true);
        // Result files consist in a single one named by default "EPHEM"
        this.getSaveResultsDialog().clearResultFileList();
        this.getSaveResultsDialog().addSingleResultFile(res.getName(), EPH_FILE_PREFIX, true);
 
        this.getSaveResultsDialog().show();
        
    }

    /**
     * Method to manage what to do before starting the computation
     */
    @Override
    protected void customPreProcessManagement() throws GFileManipulatorException {
        
        // We delete current EPHEM.txt file
        final File ephem = new File(EPH_FILE);
        if ( ephem.exists() ) {
            ephem.delete();
        }

        // We write a context file with data coming from the data panel
        GFileManipulation.writeConfig(INI_FILE, "Propagator", this.getDataPanel(), true);
        // We initialize the JavaCommandLauncher
        final String classPath = System.getProperty("java.class.path");
        this.getJavaCommandLauncher().setJavaCommand(classPath, new String[] {"propagator.BatchPropagator"});
        // We display the console above the other tabbedpanes
        this.getDataPanel().selectConsoleTab();        
    }

    /**
     * Method to manage what to do after computation
     */
    @Override
    protected void customPostProcessManagement() {
        
        // Plot results update
        if ( this.getJavaCommandLauncher().getProcessStatus() == ProcessStatus.FINISHED_NORMALY ) {
            
                // Get EPHEM file
                final File file = new File(EPH_FILE);
                
                // Update custom plots
                try {
                    this.getDataPanel().getPlots().setSelectedFile(file);
                } catch (GPlotDataReaderException e) {
                    e.printStackTrace();
                }
                
                // Reading EPHEM file
                final GPlotDataMadonaReader fileData = new GPlotDataMadonaReader();
                try {
                    fileData.load(file);
                } catch (GPlotDataReaderException e) {
                    e.printStackTrace();
                }
                
                try {
                    // Getting lon/lat columns content
                    final ArrayList<Double[]> list = new ArrayList<Double[]>(fileData.getColumns(null, new Integer[] { 4, 3 }));
                    // Update ground track
                    ArrayList<GCoordinatesData> tracks = new ArrayList<GCoordinatesData>();
                    tracks.add(new GCoordinatesData("Track", list, Color.RED, null, 180., 90.));
                    this.getDataPanel().getGroundTrack().setData(tracks);
               } catch (GPlotDataReaderException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
                    
        }
        
    }

    /**
     * Main method.
     * @param args
     * @throws GException
     * @throws PatriusException 
     */
    public static void main(String[] args) throws GException, PatriusException {
        
        // Patrius dataset initialization
        PatriusDataset.addResourcesFromPatriusDataset();
        
        final WidPropagator pan = new WidPropagator();
        pan.displayMainFrame();
 
    }
    
}
