package propagator;

import fr.cnes.genius.exception.GException;
import fr.cnes.genius.lowLevel.GPanel;
import fr.cnes.genius.main.GReadWrite;
import fr.cnes.genius.plots.GGroundPlotPanel;
import fr.cnes.genius.plots.GPlotPanel;
import fr.cnes.genius.standardapplication.GDataPanelAbstract;
import fr.cnes.genopus.exception.GPOrbitException;
import fr.cnes.genopus.exception.GPVehicleException;
import fr.cnes.genopus.forces.GPEarthPotentialForceModel.AttractionModelsEnum;
import fr.cnes.genopus.forces.GPForceModels;
import fr.cnes.genopus.orbits.GPOrbit;
import fr.cnes.genopus.vehicle.GPVehicle;
import fr.cnes.sirius.patrius.assembly.Assembly;
import fr.cnes.sirius.patrius.assembly.Vehicle;
import fr.cnes.sirius.patrius.forces.ForceModelsData;
import fr.cnes.sirius.patrius.frames.FramesFactory;
import fr.cnes.sirius.patrius.orbits.Orbit;
import fr.cnes.sirius.patrius.utils.exception.PatriusException;

/**
 * Class defining data.
 * @author goesterjf
 *
 */
public class WidPropagatorDataPanel extends GDataPanelAbstract {
    
    private final WidOrbit widOrbit;
    private final GPVehicle widVehicle;
    private final GPForceModels widForces;
    private final GPlotPanel plots;
    private final GGroundPlotPanel groundTrack;

    /**
     * Constructor.
     * @throws GException   GENIUS exception
     */
    public WidPropagatorDataPanel() throws GException {
        
        super("Data");
        
        // Creating an orbit widget 
        widOrbit =  new WidOrbit("Initial orbital parameters");
 
        // Creating a vehicle widget (only with dry mass and simple aerodynamic properties)
        widVehicle =  new GPVehicle("Vehicle characteristics", true, false, true, false, false);
 
        // Creating a force model widget (only with potential [Balmino] and atmosphere)
        AttractionModelsEnum[] attractionModelsAvailable = { AttractionModelsEnum.BALMINO };
        widForces =  new GPForceModels("Models", AttractionModelsEnum.BALMINO, attractionModelsAvailable, false, true, false, false, false, false);
        
        plots = new GPlotPanel("", "", "", 0, false, null, false);
        
        groundTrack = new GGroundPlotPanel();
        
        // Adding it in a tabbedpane
        this.addTab("Orbit", widOrbit);
        this.addTab("Vehicle", widVehicle);
        this.addTab("Forces", widForces);
        this.addTab("Plots", plots);
        this.addTab("GroundTrack", groundTrack);
        // Adding an output console
        this.addConsoleTab("Console");
        
    }

    public void clear() throws GException {
        // TODO Auto-generated method stub
    }
    
    /**
    * Private class needed because GPOrbit is not a GPanel ...
    * @author goesterjf
    *
    */
    private class WidOrbit extends GPanel implements GReadWrite {
        
        private final GPOrbit orbit;
        
        public WidOrbit ( final String label) {
            orbit = new GPOrbit(label);
        }

        public void display() throws GException { generic(); }

        @Override
        public void generic() throws GException { put(orbit); }

        public void read() throws GException { generic(); }

        public void write() throws GException { generic(); }
        
        /**
         * @return the GP orbit widget
         */
        public GPOrbit getGPOrbit() {
            return orbit;
        }
        
    }

    /**
     * Method to get the PATRIUS orbit object
     * @return PATRIUS orbit object
     * @throws GPOrbitException GENOPUS exception
     */
    public Orbit getOrbit() throws GPOrbitException {
        return widOrbit.getGPOrbit().getPatriusObject();
    }

    /**
     * Method to get the Custom PATRIUS vehicle object
     * @return the widVehicle
     * @throws GPVehicleException GENOPUS exception
     */
    public Vehicle getVehicle() throws GPVehicleException {
        return widVehicle.getPatriusObject();
    }

    /**
     * Method to get the Custom PATRIUS force models object
     * @return Custom PATRIUS force models object
     * @throws GException GENIUS exception
     * @throws PatriusException PATRIUS exception
     */
    public ForceModelsData getForces() throws GException, PatriusException {
        final Assembly assembly = getVehicle().createAssembly(FramesFactory.getCIRF());
        return widForces.getPatriusObject(assembly);
    }

    /**
     * Method to get the GPlotpanel widget 
     * @return GPlotpanel widget
     */
    public GPlotPanel getPlots() {
        return plots;
    }
    
    /**
     * Method to get the GGroundPlotPanel widget 
     * @return GGroundPlotPanel widget
     */
    public GGroundPlotPanel getGroundTrack() {
        return groundTrack;
    }
    
}
